<?php
/**
 * The plugin Base Controller
 * Defines OxyProps variables.
 *
 * @category Base
 * @package  OxyProps
 * @author   Cédric Bontems <cedric@thewebforge.dev>
 * @license  https://www.gnu.org/licenses/gpl-2.0.html  GPL v2 or later
 * @link     https://oxyprops.com                       OxyProps Website
 * @since    0.1.0
 * @since    1.5.2  Cleaned unused properties.
 */

namespace OxyProps\Inc;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * OxyProps Base Controller Class
 * Defines OxyProps variables.
 *
 * @author   Cédric Bontems <cedric@thewebforge.dev>
 * @since    1.0.0
 */
class Base_Controller {

	/**
	 * The main plugin file
	 *
	 * @var string
	 *
	 * @since  1.0.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public $oxyprops_file;

	/**
	 * The path to the plugin folder
	 *
	 * @var string
	 *
	 * @since  1.0.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public $oxyprops_path;

	/**
	 * The URL to the plugin folder
	 *
	 * @var string
	 *
	 * @since  1.0.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public $oxyprops_url;

	/**
	 * The path to the plugin file, relative to the plugins directory.
	 *
	 * @var string
	 *
	 * @since  1.0.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public $oxyprops_plugin;

	/**
	 * The plugin slug
	 *
	 * @var string
	 *
	 * @since  1.0.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public $oxyprops_slug;

	/**
	 * The plugin current version
	 *
	 * @var string
	 *
	 * @since  1.0.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public $oxyprops_version;

	/**
	 * The name of the detected active site builder
	 *
	 * @var string
	 *
	 * @since  1.0.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public $oxyprops_builder;

	/**
	 * A boolean indicating if the active site builder is supported
	 *
	 * @var boolean
	 *
	 * @since  1.4.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public $supported_builder;

	/**
	 * The plugin short name adjusted to the active site builder
	 * For use in the admin menu if too standard is too long
	 *
	 * @var string
	 *
	 * @since  1.4.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public $short_adapted_name;

	/**
	 * The plugin name adjusted to the active site builder
	 *
	 * @var string
	 *
	 * @since  1.4.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public $adapted_name;

	/**
	 * The plugin current settings
	 *
	 * @var array
	 *
	 * @since  1.4.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public $oxyprops_current_settings;

	/**
	 * $oxyprops_plugin_data.
	 *
	 * @since 1.6.0 Creation to solve deprecated dynamic variable creation.
	 * @author @cbontems
	 *
	 * @var array $oxyprops_plugin_data Plugin data from WP.
	 */
	public array $oxyprops_plugin_data;

	/**
	 * $oxyprops_text_domain.
	 *
	 * @since 1.6.0 Creation to solve deprecated dynamic variable creation.
	 * @author @cbontems
	 *
	 * @var string $oxyprops_text_domain Plugin text domain from WP data.
	 */
	public string $oxyprops_text_domain;

	/**
	 * $oxyprops_name.
	 *
	 * @since 1.6.0 Creation to solve deprecated dynamic variable creation.
	 * @author @cbontems
	 *
	 * @var string $oxyprops_name Plugin name from WP data.
	 */
	public string $oxyprops_name;

	/**
	 * BaseController class constructor
	 */
	public function __construct() {

		if ( ! function_exists( 'get_plugin_data' ) ) {
			include_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$this->oxyprops_path             = plugin_dir_path( __DIR__ );
		$this->oxyprops_slug             = plugin_basename( dirname( __DIR__ ) );
		$this->oxyprops_file             = $this->oxyprops_path . $this->oxyprops_slug . '.php';
		$this->oxyprops_plugin_data      = get_plugin_data(
			$this->oxyprops_file,
			false,
			false
		);
		$this->oxyprops_name             = $this->oxyprops_plugin_data['Name'];
		$this->oxyprops_plugin           = $this->oxyprops_slug . '/' . $this->oxyprops_slug . '.php';
		$this->oxyprops_text_domain      = $this->oxyprops_plugin_data['TextDomain'];
		$this->oxyprops_url              = plugin_dir_url( dirname( __FILE__ ) );
		$this->oxyprops_version          = $this->oxyprops_plugin_data['Version'];
		$this->oxyprops_builder          = $this->get_builder();
		$this->oxyprops_current_settings = $this->get_plugin_current_settings();
	}

	/**
	 * Gets the current plugin settings
	 *
	 * @return array Current settings
	 *
	 * @since  1.4.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public function get_plugin_current_settings() {
		$defaults = Defaults::get_instance();
		return get_option( $this->oxyprops_slug, array( 'settings' => $defaults->get_oxyprops_default_settings() ) );
	}

	/**
	 * Updates the current plugin settings
	 *
	 * @param string $key The name of the option item to be updated.
	 * @param mixed  $value The new value for the option item.
	 *
	 * @return bool True if the value was updated, false otherwise.
	 *
	 * @since  1.4.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public function update_plugin_current_settings( $key, $value ) {
		$this->oxyprops_current_settings         = $this->get_plugin_current_settings();
		$this->oxyprops_current_settings[ $key ] = $value;
		return update_option( $this->oxyprops_slug, $this->oxyprops_current_settings );
	}

	/**
	 * Checks if an option item is disabled
	 *
	 * @param string $option the option element to be checked.
	 *
	 * @return boolean
	 *
	 * @since  1.4.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	public function disabled( $option ) {
		return ! ( isset( $this->oxyprops_current_settings[ $option ] ) &&
			$this->oxyprops_current_settings[ $option ]
		);
	}

	/**
	 * Gets the builder
	 *
	 * @return string
	 *
	 * @since  1.0.0
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 */
	private function get_builder() {
		$this->supported_builder  = true;
		$this->adapted_name       = 'OxyProps';
		$this->short_adapted_name = 'OxyProps';
		$theme                    = wp_get_theme();

		if ( defined( 'CT_VERSION' ) ) {
			return 'Oxygen';
		}
		if ( 'Bricks' === $theme->name || 'Bricks' === $theme->parent_theme ) {
			$this->adapted_name       = 'BricksProps';
			$this->short_adapted_name = 'BricksProps';
			return 'Bricks';
		}
        //phpcs:disable
        // TODO: Breakdance integration.
        // if ( defined( '__BREAKDANCE_VERSION' ) ) {
        // $this->adapted_name       = 'BDProps';
        // $this->short_adapted_name = 'BDProps';
        // return 'Breakdance';
        // }
        //phpcs:enable
		$this->supported_builder = false;
		return 'Unsupported Site Buidler';
	}
}
